"use strict";
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.NdTable = exports.Table = void 0;
var output_format_sane_1 = require("./output-formatter/output-format-sane");
var output_formatters_const_1 = require("./output-formatter/output-formatters.const");
var table_builder_class_1 = require("./table-builder.class");
var table_config_class_1 = require("./table-config.class");
var table_render_data_class_1 = require("./table-render-data.class");
/**
 * Logical Table
 */
var Table = /** @class */ (function (_super) {
    __extends(Table, _super);
    function Table() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        _this._outputFormat = Table._outputFormat;
        return _this;
    }
    /**
     * Get generic table preferences
     * @returns preferences object
     */
    Table.getPreferences = function () {
        return table_config_class_1.TableConfig._preferences.value;
    };
    /**
     * Update generic table preferences
     * @param preferences updates
     * @returns this Table class
     */
    Table.setPreferences = function (preferences) {
        table_config_class_1.TableConfig._preferences.value = preferences;
        return Table;
    };
    /**
     * Sets default output format and options for all table instances that don't have their own defaults
     * @param format ascii csv html json markdown tsv utf8
     * @param options output config (speicific to the chosen format)
     * @returns the Table class
     */
    Table.setOutputFormat = function (format, options) {
        if (options === void 0) { options = {}; }
        Table._outputFormat = [(0, output_format_sane_1.outputFormatSane)(format), options];
        return Table;
    };
    /**
     * Turn 2-dimensinal data array to Table
     * @param data 2-dimensional array of values
     * @param firstRowIsHead indicates column headers
     * @param firstColumnIsHead indicates row headers
     * @returns new Table
     */
    Table.fromData = function (data, firstRowIsHead, firstColumnIsHead) {
        if (!Array.isArray(data)) {
            throw new Error('Provided data is not an array');
        }
        var table = new Table();
        var pushX = firstColumnIsHead ? 1 : 0;
        var pushY = firstRowIsHead ? 1 : 0;
        for (var y = 0 - pushY; y < data.length - pushY; y++) {
            var row = data[y + pushY];
            if (!Array.isArray(row)) {
                throw new Error('Provided data row is not an array at index ' + y);
            }
            for (var x = 0 - pushX; x < row.length - pushX; x++) {
                table.setCell(x, y, row[x + pushX]);
            }
        }
        return table;
    };
    /**
     * Sets default output format and options for the Table instance
     * @param format ascii csv html json markdown tsv utf8
     * @param options output config (speicific to the chosen format)
     * @returns this Table
     */
    Table.prototype.setOutputFormat = function (format, options) {
        if (options === void 0) { options = {}; }
        this._outputFormat = [(0, output_format_sane_1.outputFormatSane)(format), options];
        return this;
    };
    /**
     * Translate Table to ASCII string
     * @param options formatting options
     * @returns Table string
     */
    Table.prototype.toASCII = function (options) {
        return this.toString('ascii', options);
    };
    /**
     * Translate Table to comma separated values
     * @param options formatting options
     * @returns Table CSV
     */
    Table.prototype.toCSV = function (options) {
        return this.toString('csv', options);
    };
    /**
     * Translate Table to HTML
     * @param options formatting options
     * @returns Table HTML
     */
    Table.prototype.toHTML = function (options) {
        return this.toString('html', options);
    };
    /**
     * Translate Table to JSON string (array of arrays)
     * @param options formatting options
     * @returns Table JSON
     */
    Table.prototype.toJSON = function (options) {
        return this.toString('json', options);
    };
    /**
     * Translate Table to MD
     * @param options formatting options
     * @returns Table Markdown
     */
    Table.prototype.toMarkdown = function (options) {
        return this.toString('markdown', options);
    };
    /**
     * Provide a render data snapshot for the current state of the table
     * Use it for custm output formatters
     * @returns render data snapshot
     */
    Table.prototype.toRenderData = function () {
        return new table_render_data_class_1.TableRenderData(this.getSnapshot(), this._cells.configs.preferences.value);
    };
    /**
     * Created formatted table output string
     * @param format output mode (defaults to what was set by setOutputFormat() or 'utf8')
     * @param options output config (defaults to what was set by setOutputFormat())
     * @returns Table string
     */
    Table.prototype.toString = function (format, options) {
        if (options === void 0) { options = {}; }
        var defaultFormatAndOptions = (this._outputFormat || Table._outputFormat || ['utf8', {}]);
        if (format && typeof format === 'object') {
            options = format;
            format = defaultFormatAndOptions[0];
        }
        else if (!format) {
            format = defaultFormatAndOptions[0], options = defaultFormatAndOptions[1];
        }
        format = (0, output_format_sane_1.outputFormatSane)(format);
        options = options && typeof options === 'object' ? options : {};
        return output_formatters_const_1.outputFormatters[format](this.toRenderData(), options);
    };
    /**
     * Translate Table to UTF-8 string
     * @param options formatting options
     * @returns Table string
     */
    Table.prototype.toUTF8 = function (options) {
        return this.toString('utf8', options);
    };
    /**
     * Translate Table to tab separated values
     * @returns Table TSV
     */
    Table.prototype.toTSV = function (options) {
        return this.toString('tsv', options);
    };
    return Table;
}(table_builder_class_1.TableBuilder));
exports.Table = Table;
exports.NdTable = Table;
exports.default = Table;
